const express = require('express');
const cors = require('cors');
const cookieParser = require('cookie-parser');
const path = require('path');
const fs = require('fs');

const PORT = process.env.PORT || 44322;
//const PORT = process.env.PORT || (process.env.NODE_ENV === 'production' ? 44322 : 3001);

const app = express();
console.log('🚀 Starting Ayurwell CRM server...');

// Try to load backend modules
let database;

try {
  database = require('./backend/models/database');
  console.log('✅ Database loaded');
} catch (e) {
  console.log('⚠️ Database not found:', e.message);
}

// Initialize database if available
if (database && typeof database.initialize === 'function') {
  database.initialize()
    .then(() => console.log('✅ Database initialized'))
    .catch(err => console.log('⚠️ Database init failed:', err.message));
} else {
  console.log('⚠️ Database initialize function not available');
}

// Middleware (MOVED BEFORE routes)
app.use(express.json({ limit: '10mb' }));
app.use(express.urlencoded({ extended: true }));
app.use(cookieParser());
app.use(cors({
  origin: [
    'https://ayurwell.world', 
    'https://www.ayurwell.world'
  ],
  credentials: true,
  methods: ['GET', 'POST', 'PUT', 'DELETE', 'OPTIONS'],
  allowedHeaders: ['Content-Type', 'Authorization', 'Cookie']
}));

// Request logging
app.use((req, res, next) => {
  console.log(`📡 ${req.method} ${req.path}`);
  next();
});

app.get('/', (req, res) => {
  res.sendFile(path.join(__dirname, 'index.html'));
});

// ============================================================================
// API ROUTES
// ============================================================================

// POPRAWIONO: Bezpieczne ładowanie routes z try/catch
try {
  const routes = require('./backend/routes');
  app.use('/', routes);
  console.log('✅ Backend routes loaded');
} catch (e) {
  console.log('⚠️ Backend routes not found:', e.message);
  // Aplikacja będzie działać bez backend routes
}

app.get('/api', (req, res) => {
  res.json({ message: 'API works', endpoints: ['/health', '/api'] });
});

app.get('/health', (req, res) => {
  res.json({ 
    status: 'OK', 
    timestamp: new Date().toISOString(),
    port: PORT 
  });
});

// ============================================================================
// STATIC FILES & REACT
// ============================================================================

// Check if build exists, otherwise use backup/build
const buildPath = fs.existsSync(path.join(__dirname, 'build')) 
  ? path.join(__dirname, 'build')
  : path.join(__dirname, 'backup/build');

app.use('/crm', express.static(buildPath));
app.use('/static', express.static(path.join(__dirname, 'static')));
app.use('/public', express.static(path.join(__dirname, 'public')));
app.use('/uploads', express.static(path.join(__dirname, 'uploads')));
app.use(express.static(__dirname));

// ============================================================================
// FALLBACK ROUTES
// ============================================================================

// CRM React Router fallback
app.get('/crm*', (req, res) => {
  console.log('🔄 CRM route:', req.path);
  const crmIndexPath = path.join(buildPath, 'index.html');
  res.sendFile(crmIndexPath, (err) => {
    if (err) {
      console.error('❌ CRM index.html not found');
      res.status(404).json({ 
        message: 'CRM app not found',
        error: 'CRM_BUILD_NOT_FOUND'
      });
    }
  });
});

// Final fallback
app.get('*', (req, res) => {
  console.log('🔄 Final fallback:', req.path);
  res.status(404).json({ 
    message: 'Page not found',
    requested_path: req.path,
    available_routes: ['/', '/health', '/api', '/crm']
  });
});

// Error handler
app.use((err, req, res, next) => {
  console.error('🚨 Server Error:', err);
  res.status(500).json({
    message: 'Internal server error',
    error: process.env.NODE_ENV === 'development' ? err.message : 'INTERNAL_ERROR'
  });
});

// ============================================================================
// START SERVER
// ============================================================================

app.listen(PORT, '0.0.0.0', () => {
  console.log('🚀 ===================================');
  console.log(`✅ Ayurwell CRM running on port ${PORT}`);
  console.log(`🔗 Test: https://ayurwell.world/api-test`);
  console.log(`📊 API: https://ayurwell.world/api`);
  console.log(`💊 Health: https://ayurwell.world/health`);
  console.log('🚀 ===================================');
});